#include <Core/Core.h>
#include <Job/Job.h>

using namespace Upp;

class Client : public Job<String> {
public:
	Client&	Blocking(bool b = true)	{ blocking = b; return *this; }
	String	Request(const String& host, int port);
	
private:
	String	Run(Event<>&& cmd);
	bool	blocking = true;
};

String Client::Run(Event<>&& cmd)
{
	Start(pick(cmd));
	if(blocking) Finish();
	return blocking && !IsError() ? GetResult() : GetErrorDesc();
}

String Client::Request(const String& host, int port)
{
	auto cmd = [=]{
		TcpSocket	socket;
		auto& output = Job<String>::Data(); // This method allows referential access to the data of respective job.
		output = Format("Client #%d: ", GetWorkerId());
		
		INTERLOCKED {	Cout() << output << "Starting...\n"; }
		
		if(socket.Timeout(10000).Connect(host, port))
			output.Cat(socket.GetLine());
		if(socket.IsError())
			throw JobError(socket.GetError(), socket.GetErrorDesc());
	};
	return Run(cmd);
}

CONSOLE_APP_MAIN
{
	// Well known FTP servers.
	const String host1 = "test.rebex.net";
	const String host2 = "ftp.uni-erlangen.de";
	
	// Note that all Job instances are scope bound. Job-based objects
	// will call Job::Finish() when they go out of scope (get destroyed).
	Client c1, c2, c3;
	Job<int> j1;
	Array<Client> clients;
	clients.SetCount(5);
	
	// Requesting in a simple, blocking way.
	{
		Cout() << "----- Processing individual blocking requests...\n";
		Cout() << c1.Request(host1, 21) << '\n';
		Cout() << c2.Request(host2, 21)	<< '\n';
	}
	
	// Reuse workers and make requests in a simple, non-blocking way.
	{
		Cout() << "----- Processing individual non-blocking requests...\n";
		// We can "clear" the data. Useful especially if the data is "picked".
		c1.Clear();
		c2.Clear();

		c1.Blocking(false).Request(host1, 21);
		c2.Blocking(false).Request(host2, 21);

		while(!c1.IsFinished() || !c2.IsFinished())
			;
		if(c1.IsError()) Cerr() << c1.GetErrorDesc() << '\n';
		else Cout() << ~c1 << '\n';
		
		if(c2.IsError()) Cerr() << c2.GetErrorDesc() << '\n';
		else Cout() << ~c2 << '\n';

	}
	
	// Use a simple batch of non-blocking requests.
	{
		Cout() << "----- Processing a set of non-blocking requests...\n";
		for(auto& cl : clients)
			cl.Blocking(false).Request(host1, 21);
		while(!clients.IsEmpty()) {
			for(int i = 0; i < clients.GetCount(); i++) {
				auto& cl = clients[i];
				if(cl.IsFinished()) {
					Cout() << (!cl.IsError() ? cl.GetResult() : cl.GetErrorDesc()) << '\n';
					clients.Remove(i);
					break;
				}
			}
		}
	}

}

